/*____________________________________________________________________________
	Copyright (C) 1997 Network Associates Inc. and affiliated companies.
	All rights reserved.
	
	$Id: SpaceEdit.c,v 1.7 2001/02/21 02:07:47 wjb Exp $
____________________________________________________________________________*/

#include "stdio.h"
#include "stdlib.h"
#include "windows.h"
#include "pgpKeys.h"
#include "pgpConfig.h"
#include "pgpErrors.h"
#include "pgpEncode.h"
#include "pgpUtilities.h"
#include "pflPrefTypes.h"
#include "pgpPubTypes.h"
#include "pgpPassphraseUtils.h"
#include "SpaceEdit.h"

typedef struct _SE
{
	HWND hwndEdit;
	BOOL bHideTyping;
	BOOL bShowWarning;
	WNDPROC wpOrigProc;
	DWORD dwQuality;
	unsigned char szRealText[256];
	int iNumSpaces[256];
	int iLen;
	USERCALLBACK UserCallback;
	void *pUserValue;
} SE,*SEP;

//___________________________
//
// Secure memory allocation routines
//

VOID* 
SEsecAlloc (PGPContextRef context, UINT uBytes) 
{
	PGPMemoryMgrRef	memmgr;

	memmgr =  PGPPeekContextMemoryMgr (context);
	return (PGPNewSecureData (memmgr, uBytes, 0));
}


VOID 
SEsecFree (VOID* p) 
{
	if (p) {
		FillMemory ((char *)p,lstrlen((char *)p), '\0');
		PGPFreeData ((char *)p);
	}
}

void CheckCaps(SEP sep)
{
	if (GetKeyState(VK_CAPITAL) & 1)
	{
		sep->bShowWarning=TRUE;

		if(sep->UserCallback)
			(sep->UserCallback)(sep->hwndEdit,sep->pUserValue);
	}
	else
	{
		sep->bShowWarning=FALSE;

		if(sep->UserCallback)
			(sep->UserCallback)(sep->hwndEdit,sep->pUserValue);
	}
}

static void sCalculateEstimatedQuality(SEP sep)
{
	INT		i;

	// "de-obfuscate" the string for computing quality
	for(i=0; i<sep->iLen ;i++)
		sep->szRealText[i]^=sep->iNumSpaces[i];

	sep->dwQuality=pgpEstimatePassphraseQuality(sep->szRealText);

	// "re-obfuscate" the string
	for(i=0; i<sep->iLen ;i++)
		sep->szRealText[i]^=sep->iNumSpaces[i];
}

//	______________________________________
//
//  Passphrase edit box subclass procedure

LRESULT APIENTRY 
SESubclassProc (
		HWND	hWnd, 
		UINT	uMsg, 
		WPARAM	wParam, 
		LPARAM	lParam) 
{
	LRESULT				lResult;
	SEP					sep;

	sep=(SEP)GetWindowLong (hWnd, GWL_USERDATA);
	
	switch (uMsg) 
	{
		case WM_SETFOCUS :
		{
			SendMessage (hWnd, EM_SETSEL, 0xFFFF, 0xFFFF);
			break;
		}

		case WM_LBUTTONDOWN:
		{
			SetFocus(hWnd);
			return 0;
		}

		case WM_KEYUP:
		{
			CheckCaps(sep);
			break;
		}

		case WM_RBUTTONDOWN:
		case WM_LBUTTONDBLCLK :
		case WM_MOUSEMOVE :
		case WM_COPY :
		case WM_CUT :
		case WM_PASTE :
		case WM_CLEAR :
			return TRUE;

		case WM_KEYDOWN :
		{
			switch (wParam) 
			{
				case VK_HOME :
				case VK_END :
				case VK_UP :
				case VK_DOWN :
				case VK_LEFT :
				case VK_RIGHT :
				case VK_NEXT :
				case VK_PRIOR :
					return TRUE;
			}
			break;
		}

		case WM_GETTEXT :
		{
			return 0;
		}

		case WM_CHAR :
		{
			unsigned char chCharCode;
			int strsize;
			int index;

			strsize=sep->iLen;
			chCharCode = (unsigned char) wParam;

			if(chCharCode==8)
			{
				strsize--;

				// Backspace it for us
				if(strsize>=0)
				{
					sep->szRealText[strsize]=0;

					if(sep->bHideTyping)
					{
						for(index=0;index<sep->iNumSpaces[strsize];index++)
						{
							lResult = CallWindowProc (sep->wpOrigProc, 
								hWnd, uMsg, wParam, lParam); 
						}
					}
					else
					{
						lResult = CallWindowProc (sep->wpOrigProc, 
							hWnd, uMsg, wParam, lParam); 
					}

					sep->iLen--;

					sCalculateEstimatedQuality (sep);

					if(sep->UserCallback)
						(sep->UserCallback)(hWnd,sep->pUserValue);
				}

				return 0;
			}

			if(strsize==255)
				return 0;

			if(iscntrl(chCharCode))
				return 0;

			// This is XOR'ed to obsfucate, just in case something
			// "really bad" happens. The memory should be locked,
			// cleared and freed on exit however.
			sep->szRealText[strsize]=chCharCode^sep->iNumSpaces[strsize];
			sep->iLen++;

			if(sep->bHideTyping)
			{
				// Set to space so we can output random amount
				wParam=' ';

				for(index=0;index<sep->iNumSpaces[strsize];index++)
				{
					lResult = CallWindowProc (sep->wpOrigProc, 
						hWnd, uMsg, wParam, lParam); 
				}
			}
			else
			{
				lResult = CallWindowProc (sep->wpOrigProc, 
					hWnd, uMsg, wParam, lParam); 
			}

			sCalculateEstimatedQuality (sep);

			if(sep->UserCallback)
				(sep->UserCallback)(hWnd,sep->pUserValue);

			return lResult;
		}
	}
    return CallWindowProc (sep->wpOrigProc, 
		hWnd, uMsg, wParam, lParam); 
} 

VOID
SEInit(PGPContextRef context,HWND hwndEdit,
	   BOOL bHideTyping,USERCALLBACK UserCallback,
	   void *pUserValue)
{
	SEP sep;
	int index;

	sep=SEsecAlloc(context,sizeof(SE));
	memset(sep,0x00,sizeof(SE));
	sep->hwndEdit=hwndEdit;

	SetWindowLong(hwndEdit,GWL_USERDATA,(LONG)sep);

	sep->wpOrigProc = (WNDPROC) 
		SetWindowLong(hwndEdit,GWL_WNDPROC,(LONG)SESubclassProc);

	sep->bHideTyping=bHideTyping;
	sep->iLen=0;
	sep->UserCallback=UserCallback;
	sep->pUserValue=pUserValue;

	CheckCaps(sep);

	for(index=0;index<256;index++)
	{
		sep->iNumSpaces[index]=(4*rand())/RAND_MAX+1;
	}

	srand( (unsigned)time( NULL ) );
}

VOID 
SEDestroy(HWND hwndEdit)
{
	SEP sep;

//	MessageBox(NULL,"DESTROYING","MYSELF",MB_OK);

	sep=(SEP)GetWindowLong (hwndEdit, GWL_USERDATA);

	if(sep!=0)
	{
		SetWindowLong(hwndEdit,GWL_WNDPROC,(LONG)sep->wpOrigProc);
		SetWindowLong(hwndEdit,GWL_USERDATA,(LONG)0);

		memset(sep,0x00,sizeof(SE));
		SEsecFree(sep);
	}
}


//	__________________
//
//	Wipe edit box clean

VOID 
SEWipeEditBox (HWND hwndEdit)
{
	SEP sep;
	char szNull[1];
	int index;

	sep=(SEP)GetWindowLong (hwndEdit, GWL_USERDATA);

	if(sep!=0)
	{
		sep->dwQuality=0;
		sep->iLen=0;
		memset(sep->szRealText,0x00,256);
		for(index=0;index<256;index++)
		{
			sep->iNumSpaces[index]=(4*rand())/RAND_MAX+1;
		}

		szNull[0]=0;
		SendMessage(hwndEdit,WM_SETTEXT,0,(LPARAM)szNull);
	}
}

VOID 
SEGetText(HWND hwndEdit,char *copyto)
{
	SEP sep;
	int index;

	sep=(SEP)GetWindowLong (hwndEdit, GWL_USERDATA);

	for(index=0;index<sep->iLen;index++)
	{
		copyto[index]=sep->szRealText[index]^sep->iNumSpaces[index];
	}

	copyto[sep->iLen]=0;
}

DWORD
SEGetTextLength(HWND hwndEdit)
{
	SEP sep;

	sep=(SEP)GetWindowLong (hwndEdit, GWL_USERDATA);

	return sep->iLen;
}

DWORD
SEGetTextQuality(HWND hwndEdit)
{
	SEP sep;

	sep=(SEP)GetWindowLong (hwndEdit, GWL_USERDATA);

	return sep->dwQuality;
}

BOOL
SEGetShowWarning(HWND hwndEdit)
{
	SEP sep;

	sep=(SEP)GetWindowLong (hwndEdit, GWL_USERDATA);

	return sep->bShowWarning;
}

VOID 
SEChangeHideTyping (HWND hwndEdit, BOOL bHideTyping)
{
	SEP sep;
	char szSpace[256*4];
	char szPass[256];
	int strsize;
	int i,j,index,textlen;

	sep=(SEP)GetWindowLong (hwndEdit, GWL_USERDATA);

	if(bHideTyping==sep->bHideTyping)
		return;

	sep->bHideTyping=bHideTyping;

	if(bHideTyping)
	{
		strsize=sep->iLen;
		index=0;
		memset(szSpace,0x00,256*4);

		for(i=0;i<strsize;i++)
		{
			for(j=0;j<sep->iNumSpaces[i];j++)
			{
				szSpace[index]=' ';
				index++;
			}
		}

		SendMessage(hwndEdit,WM_SETTEXT,0,(LPARAM)szSpace);
	}
	else
	{
		for(i=0;i<sep->iLen;i++)
		{
			szPass[i]=sep->szRealText[i]^sep->iNumSpaces[i];
		}

		szPass[sep->iLen]=0;

		SendMessage(hwndEdit,WM_SETTEXT,0,(LPARAM)szPass);

		memset(szPass,0x00,256);
	}

	textlen=SendMessage(hwndEdit,WM_GETTEXTLENGTH,0,0);
	SendMessage(hwndEdit,EM_SETSEL,(WPARAM)textlen,(LPARAM)textlen);
}

/*__Editor_settings____

	Local Variables:
	tab-width: 4
	End:
	vi: ts=4 sw=4
	vim: si
_____________________*/

